rm(list=ls())
#' 9017 Online Panels Benchmarking Study SRM Paper
#' Analysis syntax
#'
#' @author  Dina Neiger
#' @version 20220112
#'
#'
#'
#' @input   
#' Sigtest file from Step 5
#' Benchmark data (inputs directory)
#' 
#' 
#' @output
#' Significance tests for comparison of unweighted estimates with benchmark (Table C1) and pairwise comparisons  (Table C3)
#' 
#' 
#' This syntax is structured in 3 parts:
#' ############Part 1: Calculation of test statistics
#' Two types of test statistics are calculated:
#'       - modal % for substantive variables 
#'       - absolute error (difference between the modal value and the corresponding benchmark)
#'       - final post-stratification weight is used (weight1)
#'      
#'  
#'  #############Part 2: Using bootstrap algorithm to calculate standard errors for the test statistics
#'  This method is used to ensure consistent method for standard error calculations across
#'  probability and non-probability surveys
#'  
#'  #############Part 3: Significance testing
#'  3a) Use t-test to  compare each substantive measure's modal response with the corresponding benchmark
#'      Census benchmarks are treated as true values
#'      Standard Errors for the survey benchmarks sourced from the respective sources
#'      
#'  3b) Use t-test to undertake pairwise comparisons of absolute errors between 8 surveys 
#'  
#'  The methodology is based on Yeager et al "Public Opinion Quarterly" V75, No4, pp709-747 and online supplement
#'      
#' 


options(java.parameters = "-Xmx6096m")


library("plyr")
library("xlsx")
library(reshape2)
library("foreign")

#install.packages("survey")
#install.packages("boot")

library("survey")
library("boot")

# Date formats for xlsxssf
options(xlsx.date.format="dd/MM/yyyy")
options(xlsx.datetime.format="dd/MM/yyyy")
# Windows stub
Z_PATH <- "Z:/"

setwd("Z:/Research Papers and Presentations/SRM article/Submission syntax")

WRK_DIR <- setwd("Z:/Research Papers and Presentations/SRM article/Submission syntax")

INP_DIR <- paste0(WRK_DIR,"/Inputs/")
OUT_DIR <- paste0(WRK_DIR, "/Outputs/")



#load sigdata dataset
load(paste0(OUT_DIR, "S5-udemdata.RData"))


### read var names and labels for significance testing
load(paste0(INP_DIR, "dvarnames.RData"))

#read benchmarks
load(paste0(INP_DIR, "dbmark_data.RData"))


sigtest <- function(var_ds,bmark_data,check_yes=1){

#function for calculation of statistic and absolute error calculation to be used in the boot (resampling) function
stat_fn<-function(data,var, bmrk, indices=1:nrow(data)){
  d<-data[indices,]
  if(var %in% "d16_4w1") d <- d[d$d16_base %in% 1,]
  stat   <- 100*sum(d[,var])/nrow(d) 
  abserr <- abs(stat-bmrk)  
  base   <- nrow(d)
  return(c(stat,abserr,base))
}

if(check_yes==1){
#checking code for estimate calculation
stat_fn(dsigdata_unweighted[dsigdata_unweighted$surtype.l=="2ABS",],"d16_4w1",bmark_data[bmark_data$varname %in% "d16" ,"bmrk"])
stat_fn(dsigdata_unweighted[dsigdata_unweighted$surtype.l=="2ABS",],"atsi_29w1",bmark_data[bmark_data$varname %in% "atsi","bmrk"])

#checking code for bootstrap function
   set.seed(987)
   x1 <- boot(data=dsigdata_unweighted[dsigdata_unweighted$surtype.l=="2ABS",], R=1000,
                  statistic=stat_fn,var="d16_4w1",bmrk=bmark_data[bmark_data$varname %in% "d16","bmrk"])
   print(x1)
   sd(x1$t[,1])
   sd(x1$t[,2])





   set.seed(987)
   x1 <- boot(data=dsigdata_unweighted[dsigdata_unweighted$surtype.l=="2ABS",], R=1000,
           statistic=stat_fn,var="atsi_29w1",bmrk=bmark_data[bmark_data$varname %in% "atsi","bmrk"])
   print(x1)
   sd(x1$t[,1])
   sd(x1$t[,2])

}
#create dataset of estimates, std errors, abserrors for each variable and each survey
nrow(var_ds)
est_data <- NULL
rown<-1
set.seed(987)
for(i in 1:nrow(var_ds)){
  for(j in 1:length(unique(dsigdata_unweighted$surtype))){
   est_data$varname[rown]    <- var_ds$varname[i]
   est_data$wvarname[rown]   <- var_ds$wvarname[i]
   est_data$surtype[rown]    <- j
   est_data$stat[rown]       <- stat_fn(dsigdata_unweighted[dsigdata_unweighted$surtype==j,],var_ds$wvarname[i],bmark_data[i,"bmrk"])[1]
   est_data$abserr[rown]     <- stat_fn(dsigdata_unweighted[dsigdata_unweighted$surtype==j,],var_ds$wvarname[i],bmark_data[i,"bmrk"])[2]
   secalc                    <- boot(data=dsigdata_unweighted[dsigdata_unweighted$surtype==j,], R=1000,
                                      statistic=stat_fn,var=var_ds$wvarname[i],
                                      bmrk=bmark_data[i,"bmrk"])
   est_data$stat_se[rown]   <- sd(secalc$t[,1])
   est_data$stat_base[rown] <- nrow(dsigdata_unweighted[dsigdata_unweighted$surtype==j,])
   if(var_ds$varname[i]=="d16") est_data$stat_base[rown] <- nrow(dsigdata_unweighted[dsigdata_unweighted$surtype==j & dsigdata_unweighted$d16_base %in% 1,])
   est_data$abserr_se[rown] <- sd(secalc$t[,2])
   rown                     <- rown+1
   
  }
  est_data$rank[est_data$wvarname %in% var_ds$wvarname[i]]<-rank(est_data$abserr[est_data$wvarname %in% var_ds$wvarname[i]])
  est_data$ave_abserr[est_data$wvarname %in% var_ds$wvarname[i]]<-ave(est_data$abserr[est_data$wvarname %in% var_ds$wvarname[i]])
  est_data$sd_abserr[est_data$wvarname %in% var_ds$wvarname[i]] <-sd(est_data$abserr[est_data$wvarname %in% var_ds$wvarname[i]])
}
est_data<-as.data.frame(est_data,stringsAsFactors=F)
#merge in surtype lable and benchmark data for calculation of significance
est_data <- merge(est_data,unique(dsigdata_unweighted[,c("surtype","surtype.l")],by="surtype"))
est_data <- merge(est_data,bmark_data,by="varname")

#calculate t-test for benchmark comparison
#sample benchmark
  est_data$S         <- sqrt((est_data$stat_base*(est_data$stat_base-1)* est_data$stat_se*est_data$stat_se+
                            est_data$bmrk_base*(est_data$bmrk_base-1)* est_data$bmrk_se*est_data$bmrk_se)/
                             (est_data$stat_base+est_data$bmrk_base-2))
  est_data$t.value   <- (est_data$stat-est_data$bmrk)/(est_data$S*sqrt(1/est_data$stat_base + 1/est_data$bmrk_base))
  est_data$p.value   <- round(2*pt(-abs(est_data$t.value),df=(est_data$stat_base+est_data$bmrk_base-2)),5)
#census benchmark
  est_data$t.value.c   <- (est_data$stat-est_data$bmrk)/(est_data$stat_se)
  est_data$p.value.c   <- round(2*pt(-abs(est_data$t.value.c),df=(est_data$stat_base-1)),5)

#assign t and pvalue

est_data$t.value[est_data$bmrk_se %in% 0] <- est_data$t.value.c[est_data$bmrk_se %in% 0]
est_data$p.value[est_data$bmrk_se %in% 0] <- est_data$p.value.c[est_data$bmrk_se %in% 0]


est_data$notsig05  <-""
est_data$notsig05[est_data$p.value>=0.05] <-"yes"
est_data$notsig01  <-""
est_data$notsig01[est_data$p.value>=0.01] <-"yes"

    
est_data <- as.data.frame(est_data,stringsAsFactors=F)[with(est_data, order(varlabel, surtype.l)),]
est_data_out <- est_data[,c("varlabel","varname","catname","surtype.l","stat_base","stat","bmrk",
                            "notsig05","notsig01","p.value",
                            "abserr","rank","ave_abserr","sd_abserr",
                            "bmrk_base","stat_se","bmrk_se","S","t.value","surtype","abserr_se","wvarname")]
    

chn1 <- names(est_data) %in% names(est_data_out)
names(est_data)[!chn1]

#summary by survey
survey_summary <- ddply(est_data_out,.(surtype.l),summarize,
                        ave_abserr = sum(abserr)/length(abserr),
                        no_notsig05  = sum(notsig05=="yes"),
                        no_notsig01  = sum(notsig01=="yes"),
                        no_r1      = sum(rank==1),
                        no_r2      = sum(rank==2),
                        no_r3      = sum(rank==3),
                        no_r6      = sum(rank==6),
                        no_r7      = sum(rank==7),
                        no_r8      = sum(rank==8)
                        )



#do pair wise comparisons of the average abs error
#function for calculation of statistic and absolute error calculation to be used in the boot (resampling) function
aerr_fn<-function(data, indices){
  d<-data[indices,]
  aved <- NULL
  base <- NULL
  for(i in 1:nrow(var_ds)){
    if(var_ds$var[i] %in% "d16_4w1") d <- d[d$d16_base %in% 1,]
    aved[i]<-stat_fn(d,var_ds$wvarname[i],bmark_data[i,"bmrk"])[2]
        }
  return(mean(aved))
  
}
#checking code for estimate calculation
aerr_fn(dsigdata_unweighted[dsigdata_unweighted$surtype.l %in% "2ABS",])
#calculate standard errors for average abs error

aerr_data <- NULL
set.seed(987)
for(j in 1:length(unique(dsigdata_unweighted$surtype))){
    aerr_data$surtype[j]      <- j
    aerr_data$surtype.l[j]<- unique(dsigdata_unweighted$surtype.l[dsigdata_unweighted$surtype==j])
    #keep base as the number of in scope obs for each survey type (ignore rebasing)
    aerr_data$base[j]         <- nrow(dsigdata_unweighted[dsigdata_unweighted$surtype==j,])
    aerr_data$aerr_stat[j]    <- aerr_fn(dsigdata_unweighted[dsigdata_unweighted$surtype %in% j,])
    aerr_secalc               <- boot(data=dsigdata_unweighted[dsigdata_unweighted$surtype %in% j,], R=1000,statistic=aerr_fn)
    aerr_data$aerr_se[j]      <- sd(aerr_secalc$t[,1])
    }
aerr_data <- as.data.frame(aerr_data,stringsAsFactors=F)[with(aerr_data, order(surtype.l)),]


#pairwise significance testing
pairw_m <- matrix("", nrow=length(unique(dsigdata_unweighted$surtype))+1,ncol=length(unique(dsigdata_unweighted$surtype)))
colnames(pairw_m) <- unique(aerr_data$surtype.l)
rownames(pairw_m) <- c("ave error",unique(aerr_data$surtype.l))
pairw_m

for(i in 1:nrow(pairw_m)){
  if(i==1) pairw_m[1,]<-format(round(aerr_data$aerr_stat,4),nsmall=4)
  else{
  rows <- i-1
  for(cols in 1:ncol(pairw_m)){
       #calculate the difference between average errors
       pairw_m[i,cols] <- format(round(aerr_data$aerr_stat[rows]-aerr_data$aerr_stat[cols],4),nsmall=4)
       #calculate p-value
       S         <- sqrt((aerr_data$base[rows]*(aerr_data$base[rows]-1)*aerr_data$aerr_se[rows]*aerr_data$aerr_se[rows]+
                          aerr_data$base[cols]*(aerr_data$base[cols]-1)*aerr_data$aerr_se[cols]*aerr_data$aerr_se[cols])/
                         (aerr_data$base[rows]+aerr_data$base[cols]-2)
                         )
       t.value   <- (aerr_data$aerr_stat[rows]-aerr_data$aerr_stat[cols])/
                    (S*sqrt(1/aerr_data$base[rows] + 1/aerr_data$base[cols]))
       p.value   <- round(2*pt(-abs(t.value),df=(aerr_data$base[rows]+aerr_data$base[cols]-2)),5)
       #add significance symbol +p<.10, *p<.05, **p<.01, ***p<.001
       sig.sym   <-""
       if(p.value<0.10) sig.sym  <-"+"
       if(p.value<0.05) sig.sym  <-"*"
       if(p.value<0.01) sig.sym  <-"**"
       if(p.value<0.001) sig.sym <-"***"
       pairw_m[i,cols] <- paste(pairw_m[i,cols],sig.sym)
       if(rows==cols) pairw_m[i,cols]="-"
     }
}
}
pairw_m
list_out <- list(survey_summary,est_data_out, pairw_m, aerr_data)
return (list_out)

}


#include income
sv <- unique(varnames_ds$varname) %in% bmark_data_d$varname 
table(sv)
varnames_ds$varname[!sv]

#Please note due to a bug in the loop inside the sigtest function, the code won't work unless bmark_data_d is sorted in the order of the input file supplied

dm_out <- sigtest(varnames_ds[sv ,],bmark_data_d,1)
dm_out[1]
dm_out[3]

#save benchmark comparisons
write.xlsx2(dm_out[1],paste(OUT_DIR,"dem_benchmark_comp_sigtest_13_unweighted.xlsx", sep=""),sheetName="survey summary", append=FALSE,row.names = FALSE,showNA=FALSE)
write.xlsx2(dm_out[2],paste(OUT_DIR,"dem_benchmark_comp_sigtest_13_unweighted.xlsx", sep=""),sheetName="var by survey",append=TRUE, row.names = FALSE,showNA=FALSE)

#save pairwise comparisons
write.xlsx2(dm_out[3],paste(OUT_DIR,"dem_benchmark_comp_sigtest_13_unweighted.xlsx", sep=""),sheetName="pairwise comp",append=TRUE, row.names = TRUE,showNA=FALSE)
write.xlsx2(dm_out[4],paste(OUT_DIR,"dem_benchmark_comp_sigtest_13_unweighted.xlsx", sep=""),sheetName="pairwise data",append=TRUE, row.names = FALSE,showNA=FALSE)




